/* 
 * sigqueue.c
 * Передає сам собі й перехоплює сигнали SIGUSR1, SIGRTMIN. При обробці
 * сигналів аналізує й виводить додаткову інформацію, передану разом із 
 * ними (про джерело й причину відправлення, та ін.).
 * Ілюструє порядок відправлення як звичайних сигналів так і сигналів
 * реального часу, а також порядок отримання додатковї інформації про
 * сигнали, відправлені задачами (сигнали "штучного" походження).
 * Застосовує функції sigaction(), kill(), sigqueue().
 *
 */

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>


/* Обробник сигналів SIGUSR1, SIGRTMIN. 
   В ньому можна використовувати небезпечні в контексті асинхронної
   обробки сигналів функції, оскільки він перериває або kill(), або
   sigqueue(), які є безпечними. */
void sig_handler(int signum, siginfo_t *info, void *context)
{
        printf("Номер сигналу: %d.\n", info->si_signo);
        printf("Процес-відправник: %ld.\n", (long) info->si_pid);
        printf("Реальний власник-користувач процесу-відправника: %ld.\n",
                                                (long) info->si_uid);
        switch (info->si_code) {
        case SI_USER:
                printf("Сигнал було відправлено функцією kill() або їй"
                                                        " подібною.\n");
                break;
        case SI_QUEUE:
                printf("Сигнал було відправлено функцією sigqueue(),"
                        " разом із сигналом була відправлена величина"
                        " %d.\n", info->si_value.sival_int);
                break;
        case SI_TIMER:
                printf("Сигнал було відправлено внаслідок спрацювання"
                        " таймера, разом із сигналом була відправлена"
                        " величина %d.\n", info->si_value.sival_int);
                break;
        case SI_ASYNCIO:
                printf("Сигнал було відправлено внаслідок завершення"
                        " асинхронної операції введення/виведення,"
                        " разом із сигналом була відправлена величина"
                        " %d.\n", info->si_value.sival_int);
                break;
        case SI_MESGQ:
                printf("Сигнал було відправлено внаслідок прибуття нового"
                        " повідомлення в чергу повідомлень, разом із"
                        " сигналом була відправлена величина %d.\n",
                        info->si_value.sival_int);
                break;
        default:
                printf("Причина/спосіб відправлення сигналу не"
                                                        " розпізнані.\n");
        }
}

int main()
{
        struct sigaction act;
        union sigval val;

        /* Визначає нову диспозицію для сигналів SIGUSR1, SIGRTMIN. */
        memset(&act, 0, sizeof(act));
        act.sa_flags = SA_SIGINFO;
        act.sa_sigaction = sig_handler;
        if (sigaction(SIGUSR1, &act, NULL) != 0) {
                fprintf(stderr, "Помилка при спробі встановити диспозицію"
                                " сигналу SIGUSR1: %s\n",
                                strerror(errno));
                exit(EXIT_FAILURE);
        }
        if (sigaction(SIGRTMIN, &act, NULL) != 0) {
                fprintf(stderr, "Помилка при спробі встановити диспозицію"
                                " сигналу SIGRTMIN: %s\n",
                                strerror(errno));
                exit(EXIT_FAILURE);
        }

        /* Передає сам собі сигнал SIGUSR1. */
        if (kill(getpid(), SIGUSR1) != 0) {
                fprintf(stderr, "Помилка при спробі передати собі сигнал"
                                " SIGUSR1: %s.\n", strerror(errno));
                exit(EXIT_FAILURE);
        }
        /* Передає сам собі сигнал SIGRTMIN, разом із ним відправляє
           число 12345. */
        val.sival_int = 12345;
        if (sigqueue(getpid(), SIGRTMIN, val) != 0) {
                fprintf(stderr, "Помилка при спробі передати собі сигнал"
                                " SIGRTMIN: %s.\n", strerror(errno));
                exit(EXIT_FAILURE);
        }

        exit(EXIT_SUCCESS);
}
